import 'dart:async';
import 'package:admin_desktop/domain/repository/bookings_repository.dart';
import 'package:admin_desktop/infrastructure/services/utils.dart';
import '../../domain/models/models.dart';
import 'booking_state.dart';
import 'package:flutter/material.dart';
import 'package:pull_to_refresh/pull_to_refresh.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';

class BookingNotifier extends StateNotifier<BookingState> {
  final BookingsFacade _bookingRepository;
  int _page = 0;

  BookingNotifier(this._bookingRepository) : super(const BookingState());

  void openTopBar() {
    bool isOpen = state.isOpen;
    isOpen = !isOpen;
    state = state.copyWith(isOpen: isOpen);
  }

  Future<void> fetchBookings({
    required BuildContext context,
    bool? isRefresh,
    RefreshController? controller,
    DateTime? startTime,
    DateTime? endTime,
    String? status,
  }) async {
    if (isRefresh ?? false) {
      controller?.resetNoData();
      _page = 0;
      state = state.copyWith(bookings: [], isLoading: true, hasMore: true);
    }
    final res = await _bookingRepository.getBookings(
      page: ++_page,
      startDate: startTime ?? state.startDate,
      endDate: endTime ?? state.endDate,
      masterId: state.selectMaster == -1 ? null : state.selectMaster,
      status: status,
    );
    res.when(success: (data) {
      List<BookingData> list = List.from(state.bookings);
      list.addAll(data.data ?? []);
      state = state.copyWith(
        isLoading: false,
        bookings: list,
        totalCount: data.meta?.total ?? 0,
        hasMore: list.length < (data.meta?.total ?? 0),
      );
      if (isRefresh ?? false) {
        controller?.refreshCompleted();
        return;
      } else if (data.data?.isEmpty ?? true) {
        controller?.loadNoData();
        return;
      }
      controller?.loadComplete();

      return;
    }, failure: (failure, status) {
      state = state.copyWith(isLoading: false);
      AppHelpers.errorSnackBar(context, text: failure);
    });
  }

  void changeTabIndex(int index, BuildContext context, {String? status}) {
    state = state.copyWith(selectTabIndex: index, status: status);

    fetchBookings(isRefresh: true, context: context, status: status);
  }

  void refresh() {
    state = state.copyWith(status: null, selectTabIndex: 0);
  }

  Future<void> deleteBooking(BuildContext context, int? id) async {
    state = state.copyWith(isLoading: true);
    final response = await _bookingRepository.deleteBooking(id);
    response.when(
      success: (success) {
        List<BookingData> list = List.from(state.bookings);
        list.removeWhere((element) => element.id == id);
        state = state.copyWith(bookings: list);
      },
      failure: (failure, status) {
        AppHelpers.errorSnackBar(
          context,
          text: failure,
        );
      },
    );
    state = state.copyWith(isLoading: false);
  }

  Future<void> addBooking(BookingData? booking) async {
    if (booking == null) return;
    List<BookingData> list = List.from(state.bookings);
    int? index;
    for (int i = 0; i < list.length; i++) {
      if (booking.id == list[i].id) {
        index = i;
      }
    }
    if (index != null) {
      list[index] = booking;
    } else {
      list.add(booking);
    }
    state = state.copyWith(bookings: list);
  }

  void changeStateIndex(int index) {
    state = state.copyWith(stateIndex: index);
  }

  void changeCalendarType({
    required int index,
    required BuildContext context,
  }) {
    DateTime? startDate;
    DateTime? endDate;
    switch (index) {
      case 0:
        startDate = DateTime.now();
        endDate = DateTime.now();
        break;
      case 1:
        startDate = DateTime.now().subtractDay(1);
        endDate = DateTime.now().addDay(1);
        break;
      case 2:
        startDate = DateTime.now().firstDayOfWeek();
        endDate = DateTime.now().lastDayOfWeek();
        break;
    }
    state = state.copyWith(
        startDate: startDate,
        endDate: endDate,
        calendarType: index,
        calendarZoom: 2.2);
    fetchBookings(
      context: context,
      isRefresh: true,
      startTime: startDate,
      endTime: endDate,
    );
  }

  void changeMaster({required int? id, required BuildContext context}) {
    state = state.copyWith(selectMaster: id ?? 0);
    fetchBookings(context: context, isRefresh: true);
  }

  void changeStatus({required String? status, required BuildContext context}) {
    state = state.copyWith(status: state.status == status ? null : status);
    fetchBookings(context: context, isRefresh: true);
    //  Navigator.pop(context);
  }

  void changeZoom({double? i}) {
    state = state.copyWith(calendarZoom: i ?? state.calendarZoom);
  }

  void changeDate(BuildContext context, {DateTime? startDate}) {
    final endDate = state.calendarType == 0
        ? startDate
        : state.calendarType == 1
            ? startDate?.addDay(2)
            : startDate?.addDay(6);
    state = state.copyWith(startDate: startDate, endDate: endDate);
    Delayed(milliseconds: 300).run(() async {
      fetchBookings(
        context: context,
        isRefresh: true,
        startTime: startDate,
        endTime: endDate,
      );
    });
  }

  void selectBooking({int? id, required int orderLength}) {
    List list = List.from(state.selectBooking);
    if (state.selectBooking.contains(id)) {
      list.remove(id);
    } else {
      list.add(id ?? 0);
    }
    state = state.copyWith(selectBooking: list);

    if (list.length == orderLength) {
      state = state.copyWith(isAllSelect: true);
    } else {
      state = state.copyWith(isAllSelect: false);
    }
  }

  void allSelectBooking(List<BookingData> orderList) {
    List list = [];
    if (!state.isAllSelect) {
      for (int i = 0; i < orderList.length; i++) {
        list.add(orderList[i].id);
      }
      state = state.copyWith(isAllSelect: true);
    } else {
      state = state.copyWith(isAllSelect: false);
    }
    state = state.copyWith(selectBooking: list);
  }
}
